/**
 * @file: mspin_appl_if_adapter.h
 *
 * @version: $Id:$
 *
 * @release: $Name:$
 *
 * mySPIN ApplInterfaceAdapter
 *
 * @component: MSPIN
 *
 * @author: Torsten Plate ICT-ADITG/SW2 tplate@de.adit-jv.com
 *
 * @copyright: (c) 2003 - 2013 ADIT Corporation
 *
 * @history
 * 0.1 TPlate Initial version
 *
 ***********************************************************************/

#ifndef MSPIN_APPL_IF_ADAPTER_H_
#define MSPIN_APPL_IF_ADAPTER_H_

#ifdef  __cplusplus
extern "C"
{
#endif

#include <stdbool.h>

#ifdef PROJECTBUILD_NISSAN_LCN2
#include <adit_typedef.h>
#else
#include <adit-system/adit_typedef.h>
#endif

// ------------------------------------------------------------------------------------------------
// Adapter API
// These are the types and functions published to the system
// ------------------------------------------------------------------------------------------------

//
// Defines
//

#ifndef MSPIN_UNUSED
#define MSPIN_UNUSED(X) (void)X;
#endif //MSPIN_UNUSED

//
// Types
//

/**
 * MSPIN_PIXEL_FORMAT
 */
typedef enum
{
    MSPIN_PIXEL_FORMAT_RGB888 = 1,   //!< MSPIN_PIXEL_FORMAT_RGB888
    MSPIN_PIXEL_FORMAT_RGB565 = 2,   //!< MSPIN_PIXEL_FORMAT_RGB565
    MSPIN_PIXEL_FORMAT_ARGB8888 = 3, //!< MSPIN_PIXEL_FORMAT_ARGB8888
    MSPIN_PIXEL_FORMAT_RGBA8888 = 4, //!< MSPIN_PIXEL_FORMAT_RGBA8888
} MSPIN_PIXEL_FORMAT;

/**
 * MSPIN_FRAME_COMPRESSION
 */
typedef enum
{
    MSPIN_FRAME_COMPRESSION_NONE = 0,           //!< MSPIN_FRAME_COMPRESSION_NONE
    MSPIN_FRAME_COMPRESSION_JPEG = 1,           //!< MSPIN_FRAME_COMPRESSION_JPEG
    MSPIN_FRAME_COMPRESSION_ZLIB = 2,           //!< MSPIN_FRAME_COMPRESSION_ZLIB
    MSPIN_FRAME_COMPRESSION_UNCOMPRESSED = 4    //!< MSPIN_FRAME_COMPRESSION_UNCOMPRESSED
} MSPIN_FRAME_COMPRESSION;

typedef enum
{
    MSPIN_SUCCESS = 0,                      // success
    MSPIN_ERROR_CONNECTION_START,           // failed to connect to the given phone (AOAP/iAP)
    MSPIN_ERROR_CORE,                       // mySPIN Core initialization failed
    MSPIN_ERROR_LAYERMANAGER,               // layer manager initialization failed
    MSPIN_ERROR_GENERAL,                    // general error
    MSPIN_ERROR_CONNECTION_TIMEOUT,         // connection timeout
    MSPIN_ERROR_CONNECTION_EOF_ZERO,        // receive function returned EOF or 0 bytes
    MSPIN_ERROR_PROTOCOL_STOP,              // protocol status change from RUN to STOP
    MSPIN_ERROR_Z_DATA_ERROR,               // error while uncompressing frame data
    MSPIN_ERROR_MEM_ALLOC,                  // memory allocation needed for operation failed
    MSPIN_ERROR_FRAME_SIZE_MISMATCH,        // uncompressed frame size != expected size (width*height*bpp)
    MSPIN_ERROR_PROTOCOL_SETUP,             // protocol setup time exceeded (check dialogue and app on phone/re-plug phone)
    MSPIN_ERROR_RECEIVER_START_FAILED,      // pthread_create() for main thread returned error
    MSPIN_ERROR_MESSAGE_PROCESSING_FAILED,  // message processing failed: NULL pointer, unknown event or memory allocation
    MSPIN_ERROR_NOT_IMPLEMENTED,            // method called is not yet implemented. Intended for future releases
    MSPIN_ERROR_INVALID_PARAMETER,          // invalid parameter
    MSPIN_ERROR_NULL_HANDLE,                // NULL handle
    MSPIN_ERROR_NOT_SUPPORTED,              // Feature not supported
    MSPIN_ERROR_NOT_READY,                  // mySPIN isn't ready to do this
    MSPIN_ERROR_JPEG,                       // JPEG error
    MSPIN_ERROR_INITIALIZATION_ABORT,       // Initialization abort
    MSPIN_ERROR_NETWORK_NOT_REACHABLE,      // Network (IP address) not reachable
    MSPIN_ERROR_CONNECTION_REFUSED,         // TCP/IP connection refused
    MSPIN_ERROR_NOT_FOUND,                  // Value not found
    MSPIN_ERROR_ALREADY_RUNNING,            // Already running, stop first
    MSPIN_ERROR_NOT_ALLOWED,                // Not allowed to do this. For example when trying to stop something from the same thread
    MSPIN_ERROR_UNKNOWN
} MSPIN_ERROR;

/**
 * MSPIN_STRINGENCODING
 */
typedef enum
{
    MSPIN_STRINGENCODING_UTF8 = 1,
    MSPIN_STRINGENCODING_ASCII_LATIN1 = 2
} MSPIN_STRINGENCODING;

/**
 * MSPIN_KEY
 */
typedef enum
{
	MSPIN_KEY_HOME,
	MSPIN_KEY_BACK,
	MSPIN_KEY_MENU,
	MSPIN_KEY_SEARCH
} MSPIN_KEY;

typedef enum
{
    MSPIN_TOUCHTYPE_DOWN = 1,
    MSPIN_TOUCHTYPE_UP = 2,
    MSPIN_TOUCHTYPE_MOVE = 3,
    MSPIN_TOUCHTYPE_CANCEL = 4
} MSPIN_TOUCHTYPE;

typedef enum
{
    MSPIN_PHONECALLSTATUS_RECEIVED = 1,
    MSPIN_PHONECALLSTATUS_STARTED,
    MSPIN_PHONECALLSTATUS_ACTIVE,
    MSPIN_PHONECALLSTATUS_ENDED,
    MSPIN_PHONECALLSTATUS_WAITING,
    MSPIN_PHONECALLSTATUS_REJECTED
} MSPIN_PHONECALLSTATUS;

typedef enum
{
    MSPIN_HOSTMSGSTRTYPE_BTMACADDRESS = 1,
    MSPIN_HOSTMSGSTRTYPE_FIRMWAREVERSION_A,
    MSPIN_HOSTMSGSTRTYPE_FIRMWAREVERSION_B,
    MSPIN_HOSTMSGSTRTYPE_FIRMWAREVERSION_C,
    MSPIN_HOSTMSGSTRTYPE_FIRMWAREVERSION_D,
    MSPIN_HOSTMSGSTRTYPE_FIRMWAREVERSION_E,
    MSPIN_HOSTMSGSTRTYPE_FIRMWAREVERSION_F,
    MSPIN_HOSTMSGSTRTYPE_BUNDLEIDIOS
} MSPIN_HOSTMSGSTRTYPE;

typedef enum
{
    MSPIN_HOSTMSGINTTYPE_SPBHUTUNERCLIENT = 65024,
    MSPIN_HOSTMSGINTTYPE_SCREENTEST = 65280,
    MSPIN_HOSTMSGINTTYPE_GETCANAMBIENTLIGHT = 65281
} MSPIN_HOSTMSGINTTYPE;

typedef enum
{
    MSPIN_CUSTOMDATA_KEYPOROFILE_NONE = 0,  ///No profiles. Disables key list profiles. This is default
    MSPIN_CUSTOMDATA_KEYPOROFILE_4KEYS = 1, ///Enables 4 keys: MSPIN_CUSTOMDATA_KEYCODE_OK, MSPIN_CUSTOMDATA_KEYCODE_BACK,
                                            ///MSPIN_CUSTOMDATA_KEYCODE_DPADUP and MSPIN_CUSTOMDATA_KEYCODE_DPADDOWN
    MSPIN_CUSTOMDATA_KEYPOROFILE_6KEYS = 2, ///Enables 6 keys: MSPIN_CUSTOMDATA_KEYCODE_OK, MSPIN_CUSTOMDATA_KEYCODE_BACK,
                                            ///MSPIN_CUSTOMDATA_KEYCODE_DPADUP, MSPIN_CUSTOMDATA_KEYCODE_DPADDOWN,
                                            ///MSPIN_CUSTOMDATA_KEYCODE_DPADLEFT,and MSPIN_CUSTOMDATA_KEYCODE_DPADRIGHT
} MSPIN_CUSTOMDATA_KEYPROFILE;

typedef enum
{
    MSPIN_CUSTOMDATA_KEYTYPE_PRESS = 1,
    MSPIN_CUSTOMDATA_KEYTYPE_RELEASE = 2,
    MSPIN_CUSTOMDATA_KEYTYPE_LONGPRESS = 3,
    MSPIN_CUSTOMDATA_KEYTYPE_CLICK = 4,
    MSPIN_CUSTOMDATA_KEYTYPE_DOUBLECLICK = 5
} MSPIN_CUSTOMDATA_KEYTYPE;

typedef enum
{
     MSPIN_CUSTOMDATA_KEYCODE_PUSHTOTALK = 1,
     MSPIN_CUSTOMDATA_KEYCODE_SCROLLUP = 2,
     MSPIN_CUSTOMDATA_KEYCODE_SCROLLDOWN = 3,
     MSPIN_CUSTOMDATA_KEYCODE_PAGEUP = 4,
     MSPIN_CUSTOMDATA_KEYCODE_PAGEDOWN = 5,
     MSPIN_CUSTOMDATA_KEYCODE_VALUEUP = 6,
     MSPIN_CUSTOMDATA_KEYCODE_VALUEDOWN = 7,
     MSPIN_CUSTOMDATA_KEYCODE_PHONELIFTRECEIVER = 8,
     MSPIN_CUSTOMDATA_KEYCODE_PHONERINGOFF = 9,
     MSPIN_CUSTOMDATA_KEYCODE_VOLUMEUP = 10,
     MSPIN_CUSTOMDATA_KEYCODE_VOLUMEDOWN = 11,
     MSPIN_CUSTOMDATA_KEYCODE_OK = 12,
     MSPIN_CUSTOMDATA_KEYCODE_CANCEL = 13,
     MSPIN_CUSTOMDATA_KEYCODE_BACK = 14,
     MSPIN_CUSTOMDATA_KEYCODE_KNOBTICKCW = 20,
     MSPIN_CUSTOMDATA_KEYCODE_KNOBTICKCCW = 21,
     MSPIN_CUSTOMDATA_KEYCODE_DPADUP = 30,
     MSPIN_CUSTOMDATA_KEYCODE_DPADDOWN = 31,
     MSPIN_CUSTOMDATA_KEYCODE_DPADLEFT = 32,
     MSPIN_CUSTOMDATA_KEYCODE_DPADRIGHT = 33
} MSPIN_CUSTOMDATA_KEYCODE;


typedef enum
{
    MSPIN_PHONEMSGSTRTYPE_BTNAME = 1,
    MSPIN_PHONEMSGSTRTYPE_BUNDLEIDIOS = 2
} MSPIN_PHONEMSGSTRTYPE;

typedef enum
{
    MSPIN_PHONEMSGINTTYPE_BATTERYLEVEL = 1,
    MSPIN_PHONEMSGINTTYPE_CARRIERSIGNALLEVEL = 2,
    MSPIN_PHONEMSGINTTYPE_MOBILECONNECTIONLEVEL = 3,
    MSPIN_PHONEMSGINTTYPE_DATETIME = 4,
    MSPIN_PHONEMSGINTTYPE_BATTERYSTATUS = 5,
    MSPIN_PHONEMSGINTTYPE_DATETIMELOCAL = 6,
    MSPIN_PHONEMSGINTTYPE_SPBHUTUNERSERVER = 65024,
    MSPIN_PHONEMSGINTTYPE_SETCANAMBIENTLIGHT = 65281
} MSPIN_PHONEMSGINTTYPE;

typedef enum
{
    MSPIN_VOICESESSION_NO_CONSTRAINT = 0,
    MSPIN_VOICESESSION_ONLY_MIC_AND_SPEAKER_TOGETHER = 1
} MSPIN_VOICESESSION_CONSTRAINT;

typedef enum
{
    MSPIN_VOICESESSION_NOT_SUPPORTED = 0,
    MSPIN_VOICESESSION_VIRTUAL_PHONE_CALL_ONLY = 1,
    MSPIN_VOICESESSION_FULL_SCO_SUPPORT = 2,
    MSPIN_VOICESESSION_PROTOCOL_TRANSPORT = 3,
    MSPIN_VOICESESSION_SECOND_ENDPOINT = 4
} MSPIN_VOICESESSION_SUPPORT;

typedef enum
{
    MSPIN_VOICESESSION_REQUESTTYPE_START_MIC_AND_SPEAKERS = 0,
    MSPIN_VOICESESSION_REQUESTTYPE_START_MIC_ONLY = 1,
    MSPIN_VOICESESSION_REQUESTTYPE_ACTIVE = 2,
    MSPIN_VOICESESSION_REQUESTTYPE_END_USER_EVENT = 3,
    MSPIN_VOICESESSION_REQUESTTYPE_END_REAL_CALL_EVENT = 4,
} MSPIN_VOICESESSION_REQUESTTYPE;

typedef enum
{
    MSPIN_VOICESESSION_STATUS_UNAVAILABLE = 0,
    MSPIN_VOICESESSION_STATUS_IDLE = 1,
    MSPIN_VOICESESSION_STATUS_START_REQUESTED = 2,
    MSPIN_VOICESESSION_STATUS_ACTIVE = 3,
    MSPIN_VOICESESSION_STATUS_END_REQUESTED = 4
} MSPIN_VOICESESSION_STATUS;

typedef enum
{
    MSPIN_VOICESESSION_STATUS_CONSTRAINT_NONE = 0,
    MSPIN_VOICESESSION_STATUS_CONSTRAINT_NOHFP = 1,
    MSPIN_VOICESESSION_STATUS_CONSTRAINT_VOICEALREADYACTIVE = 2,
    MSPIN_VOICESESSION_STATUS_CONSTRAINT_SCOTIMEOUT = 3
} MSPIN_VOICESESSION_STATUS_CONSTRAINT;

typedef enum
{
    MSPIN_NAVIGATETOTYPE_PositionOnly = 1,
    MSPIN_NAVIGATETOTYPE_PlacemarkOnly = 2,
    MSPIN_NAVIGATETOTYPE_Both = 3
} MSPIN_NAVIGATETOTYPE;

typedef enum
{
    MSPIN_LOCATIONMASK_Country       = 0x0001,
    MSPIN_LOCATIONMASK_StateRegion   = 0x0002,
    MSPIN_LOCATIONMASK_ZipCode       = 0x0004,
    MSPIN_LOCATIONMASK_City          = 0x0008,
    MSPIN_LOCATIONMASK_Street        = 0x0010,
    MSPIN_LOCATIONMASK_CrossStreet   = 0x0020,
    MSPIN_LOCATIONMASK_Housenumber   = 0x0040
} MSPIN_LOCATIONMASK;

typedef enum
{
    MSPIN_SERVERCAPABILITIES_NONE = 0,
    MSPIN_SERVERCAPABILITIES_VOICECONTROLDEPRECATED = 1, // deprecated - don't use
    MSPIN_SERVERCAPABILITIES_AUDIOHANDLING = 2,
    MSPIN_SERVERCAPABILITIES_APPSTART = 4,
    MSPIN_SERVERCAPABILITIES_ALL = 7
} MSPIN_SERVERCAPABILITIES;

typedef enum
{
    MSPIN_CLIENTCAPABILITIES_NONE = 0,
    MSPIN_CLIENTCAPABILITIES_INITIATEPHONECALL = 1,
    MSPIN_CLIENTCAPABILITIES_NAVIGATETO = 2,
    MSPIN_CLIENTCAPABILITIES_SCREENCOPY = 4, //only for internal usage, cannot be applied
    MSPIN_CLIENTCAPABILITIES_POSITIONINFORMATION = 8,
    MSPIN_CLIENTCAPABILITIES_AUDIOHANDLING = 16,
    MSPIN_CLIENTCAPABILITIES_PACKETEXTENSIONS = 32, //only for internal usage, cannot be applied
    MSPIN_CLIENTCAPABILITIES_PARTIALFRAMEUPDATE = 64,
    MSPIN_CLIENTCAPABILITIES_NOHOMEBUTTON = 128,
    MSPIN_CLIENTCAPABILITIES_REQUIRESFOCUSCONTROL = 256,
    MSPIN_CLIENTCAPABILITIES_ISTWOWHEELER = 512,
    MSPIN_CLIENTCAPABILITIES_ISOTHERVEHICLE = 1024,
    MSPIN_CLIENTCAPABILITIES_ALL = 2047
} MSPIN_CLIENTCAPABILITIES;

typedef enum
{
    MSPIN_AUDIOTYPE_IGNORE = 1,
    MSPIN_AUDIOTYPE_SILENT = 2,
    MSPIN_AUDIOTYPE_MAIN = 3,
    MSPIN_AUDIOTYPE_CHIME = 4,
    MSPIN_AUDIOTYPE_ANNOUNCEMENT = 5,
    MSPIN_AUDIOTYPE_CRITICALANNOUNCEMENT = 6
} MSPIN_AUDIOTYPE;

typedef enum
{
    MSPIN_AUDIOCONTROL_REQUEST = 1,
    MSPIN_AUDIOCONTROL_RELEASE = 5
} MSPIN_AUDIOCONTROL;

typedef enum
{
    MSPIN_AUDIOSTATUS_REJECT = 2,
    MSPIN_AUDIOSTATUS_OPEN = 3,
    MSPIN_AUDIOSTATUS_SUSPEND = 4,
    MSPIN_AUDIOSTATUS_CLOSE = 6
} MSPIN_AUDIOSTATUS;

typedef enum
{
    MSPIN_AUDIORESULTCODE_NOERROR = 0,
    MSPIN_AUDIORESULTCODE_UNKNOWNERROR = 255
} MSPIN_AUDIORESULTCODE;

typedef enum
{
    MSPIN_LOGGING_TO_NONE = 0x00,
    MSPIN_LOGGING_TO_DLT = 0x01,
    MSPIN_LOGGING_TO_TTFIS = 0x02,
    MSPIN_LOGGING_TO_STDOUT = 0x04,
    MSPIN_LOGGING_TO_FILE = 0x08,
} MSPIN_LOGGING_DESTINATION;

typedef enum
{
    MSPIN_UDP_BROADCAST_END_REASON_QUIT = 0,
    MSPIN_UDP_BROADCAST_END_REASON_SETUP_FAILED = -1,
    MSPIN_UDP_BROADCAST_END_REASON_IP_ADDRESS_FAILURE = -2,
    MSPIN_UDP_BROADCAST_END_REASON_SEND_FAILED = -3
} MSPIN_UDP_BROADCAST_END_REASON;

typedef enum
{
    MSPIN_TCP_CONNECTION_REJECTED_BY_USER = 1,              //!< Connection was closed by user (rejected)
    MSPIN_TCP_CONNECTION_MAX_CONNECTIONS_REACHED = 2,       //!< maximum number of open socket connections reached. User must first close connections before new ones can be accepted.
    MSPIN_TCP_CONNECTION_MYSPIN_SESSION_ENDED = 3,          //!< mySPIN session was ended and socket was closed by this component (call of MSPIN_DisconnectTCPClient)
    MSPIN_TCP_CONNECTION_REMOTE_END_HANG_UP = 4,            //!< Remote end closed socket (LAN connection dropped or application closed socket)
    MSPIN_TCP_CONNECTION_SOCKET_ERROR = 5,                  //!< Read/write error on socket. Could be caused by a loss of connectivity
    MSPIN_TCP_CONNECTION_OTHER_ERROR = 6,                   //!< Accept callback not registered or something failed during setup
    MSPIN_TCP_CONNECTION_TLS_CLIENT_VERIFICATION_FAILED = 7,//!< TLS only: Client verification failed
    MSPIN_TCP_CONNECTION_TLS_NO_CLIENT_CERT = 8,            //!< TLS only: Client did not sent a certificate or certificate is empty
    MSPIN_TCP_CONNECTION_TLS_ACCEPT_ERROR = 9,              //!< TLS only: TLS_Accept failed
    MSPIN_TCP_CONNECTION_TLS_ERROR = 10,                    //!< TLS only: TLS error
} MSPIN_TCP_CONNECTION_END_REASON;

typedef enum
{
    MSPIN_LAUNCHER_STATE_NOT_VISIBLE = 0,                   //!< Launcher not visible => sending home button has some mySPIN effect
    MSPIN_LAUNCHER_STATE_VISIBLE_FIRST_SCREEN = 1,          //!< Launcher visible and shows first screen  => sending home button has no effect
    MSPIN_LAUNCHER_STATE_VISIBLE_OTHER_SCREEN = 2           //!< Launcher visible but not on first screen => sending home button has some mySPIN effect
} MSPIN_LAUNCHER_STATE;

typedef enum
{
    MSPIN_APPSTARTSTATE_UNKNOWN = 0,                        //!< The default and initial value
    MSPIN_APPSTARTSTATE_INPROGRESS = 1,                     //!< The App is starting
    MSPIN_APPSTARTSTATE_SUCCESS = 2,                        //!< The App has been started successfully
    MSPIN_APPSTARTSTATE_FAILED = 128,                       //!< The App could not be started
    MSPIN_APPSTARTSTATE_NOTINSTALLED = 129,                 //!< The App could not be started as it is not yet installed on the phone, although it is possible to install it
    MSPIN_APPSTARTSTATE_NOTALLOWED = 130,                   //!< The App can not be started as it is not or no longer allowed to be started (i.e. it has been removed from whitelist)
    MSPIN_APPSTARTSTATE_NOTPAIRED = 131                     //!< The App is not yet paired. App has to be paired once before usage, user action on phone required.
} MSPIN_APP_START_STATE;

typedef enum
{
    MSPIN_APPINSTALLATIONSTATUS_UNKNOWN  = 0,               //!< Default and initial value
    MSPIN_APPINSTALLATIONSTATUS_NOTINSTALLED = 1,           //!< App is not installed, but could be installed
    MSPIN_APPINSTALLATIONSTATUS_NOTPAIRED = 2,              //!< App is not paired. App has to be paired once before usage, user action on phone required.
    MSPIN_APPINSTALLATIONSTATUS_EXECUTABLE = 3              //!< App can be used
} MSPIN_APP_INSTALLATION_STATUS;

typedef struct
{
    MSPIN_TOUCHTYPE event;
    U16 xPosition;
    U16 yPosition;
    U32 fingerId;
} MSPIN_TouchEvent_t;

typedef struct
{
    U32 width;
    U32 height;
    U32 physicalWidth;
    U32 physicalHeight;
    MSPIN_PIXEL_FORMAT format;
    MSPIN_FRAME_COMPRESSION compression;
    MSPIN_FRAME_COMPRESSION preferredCompression;
    U32 layerId;
    U32 surfaceId;
} MSPIN_FrameProperties_t;

typedef struct
{
    const char* vendor;
    const char* model;
    const char* description;
    const char* version;
    const char* uri;
    const char* serial;
} MSPIN_AccessoryPropertiesAoap_t;

typedef struct
{
    U16 validMask;
    char* Country;
    char* StateRegion;
    char* ZipCode;
    char* City;
    char* Street;
    char* CrossStreet;
    char* Housenumber;
} MSPIN_LocationStrings_t;

/**
 * @brief Identifiers used within the UDP broadcast message
 */
typedef struct
{
    S32 tcpPort;                    ///TCP port used for mySPIN connection
    const U8* manufacturer;         ///manufacturer e.g. Bosch (max 64 bytes incl. null termination)
    const U8* vendor;               ///vendor of the car (max 64 bytes incl. null termination)
    const U8* model;                ///model of the car/headunit (max 64 bytes incl. null termination)
    const U8* version;              ///version (max 12 bytes incl. null termination)
    bool tls;                       ///if set to TRUE, use TLS encryption. If set to false, an unsecured TCP/IP connection will be used. When setting to TRUE, use MSPIN_StartTLSListener. Otherwise use MSPIN_StartTCPListener
    const U8* btName;               ///Bluetooth name of the IVI. This is an optional parameter.
} MSPIN_UDP_MESSAGE_PARAMETER_t;

/**
 * @brief TLS configuration parameters
 */
typedef struct
{
    const char* certificate;        //Path and name of server's certificate file (must be '/0' terminated)
    const char* privateKey;         //Path and name of server's private key file (must be '/0' terminated)
    const char* ca;                 //Path and name of certificate authority (CA) file (must be '/0' terminated)
    bool aes256only;                //Use only AES256 encryption
    bool verifyClient;              //Verify client by retrieving client certificate and checking against CA
} MSPIN_TLS_CONFIGURATION_t;

typedef struct
{
    char* IdentifierURL;
    U32 UniqueIconID;
    U32 uncompressedDataLength;
    U8* uncompressedData;
} MSPIN_AppIconResponse_t;

typedef struct
{
    U8 NumberOfCategories;
    U32* Categories;
} MSPIN_CategoryList_t;

typedef struct
{
    char* key;
    char* value;
} MSPIN_KeyValue_t;

typedef struct
{
    U8 NumberOfEntries;
    MSPIN_KeyValue_t* Entries;
} MSPIN_KeyValueList_t;

typedef struct
{
    char* IdentifierURL;
    char* Name;
    U32 UniqueIconID;
    char* Manufacturer;
    char* Description;
    MSPIN_CategoryList_t* Categories;
    MSPIN_APP_INSTALLATION_STATUS InstStatus;
    bool IsAbleToLaunchApps;
    U8 arrLanguage[3];
} MSPIN_AppInfo_t;

typedef struct
{
    U8 NumberOfEntries;
    MSPIN_AppInfo_t** Entries;
} MSPIN_AppInfoList_t;

typedef struct mspin_context_t MSPIN_Instance_t; //mspin_context_t is defined internally

typedef S32 (*MSPIN_EAPSend)(const U8* buffer, U32 bufferLen, void *pContext);

typedef S32 (*MSPIN_EAPRead)(U8* buffer, U32 bufferLen, void *pContext);

/**
 * @brief MSPIN_OnUDPBroadcastEnd
 *
 * This callback will be called when UDP broadcasting ends. UDP broadcasting is
 * started with MSPIN_StartUDPBroadcasting
 *
 * @param reason The reason why UDP broadcasting ended
 * @param context The user context registered within MSPIN_StartUDPBroadcasting
 */
typedef void (*MSPIN_OnUDPBroadcastEnd)(MSPIN_UDP_BROADCAST_END_REASON reason, void* context);

/**
 * @brief MSPIN_OnUDPUnicastEnd
 *
 * This callback will be called when UDP unicasting ends. UDP unicasting is
 * started with MSPIN_StartUDPUnicasting
 *
 * @param reason The reason why UDP unicasting ended. It uses the same enum as
 *               UDP broadcasting
 * @param context The user context registered within MSPIN_StartUDPUnicasting
 */
typedef void (*MSPIN_OnUDPUnicastEnd)(MSPIN_UDP_BROADCAST_END_REASON reason, void* context);

/**
 * @brief MSPIN_OnAcceptIncomingConnection
 *
 * This callback will be called when a socket connection in server mode is accepted. By
 *
 * @param connectionID The connection ID to be used for further calls like MSPIN_ConnectTCPClient. The connection
 *                     ID is always positive (including 0).
 * @param ipAddr The IP address of the client who established the connection. The string is null terminated. If the IP
 *               address shall be used later, the string must be copied!
 * @param context The user context registered within MSPIN_StartTCPListener
 * @return true => accept socket connection (can be also later manually closed with MSPIN_CloseConnection or
 *                 automatically when a mySPIN session is disconnected
 *         false => deny socket connection (will be closed immediately)
 */
typedef bool (*MSPIN_OnAcceptIncomingConnection)(S32 connectionID, const U8* ipAddr, void* context);

/**
 * @brief MSPIN_OnConnectionClosed
 *
 * This callback will be called when a socket connection gets closed. The reason will be specified.
 *
 * @param connectionID The connection ID of the closed connection (will be reported for example in MSPIN_OnAcceptIncomingConnection)
 * @param reason The reason why the connection got closed
 * @param ipAddr The IP address of the client who established the socket connection
 * @param context The context registered with MSPIN_StartTCPListener
 */
typedef void (*MSPIN_OnConnectionClosed)(S32 connectionID, MSPIN_TCP_CONNECTION_END_REASON reason, const U8* ipAddr, void* context);

/**
 * @func Definition of UDP message received callback
 *
 * When registered this callback will only be triggered when a valid mySPIN UDP broadcast message is received.
 *
 * @param hostname The hostname of the mySPIN phone (normally the IP address)
 * @param portNumber The port number which can be used for connecting to mySPIN
 * @param pContext The UDP listener context registered together with this callback which might hold any user
 *                 information.
 */
typedef void (*MSPIN_UDPMessageReceived)(const U8* hostname, U32 portNumber, void *pContext);

/**
 * @func MSPIN_OnLauncherStateChange
 *
 * Definition of launcher state change callback
 *
 * @param pContext The user context registered with MSPIN_CreateInstance
 * @param launcherState The new launcher state
 */
typedef void(*MSPIN_OnLauncherStateChange)(void* pContext, MSPIN_LAUNCHER_STATE launcherState);

/**
 * @func MSPIN_OnHomeButtonUsabilityChange
 *
 * Definition of home button usability change callback
 *
 * @param pContext The user context registered with MSPIN_CreateInstance
 * @param usable When true, the home button can be used within mySPIN. When false, the home button\
 *               does not have any effect in mySPIN because we are already on first screen (of launcher)
 */
typedef void(*MSPIN_OnHomeButtonUsabilityChange)(void* pContext, bool usable);

/**
 * @func MSPIN_OnAppListChanged
 *
 * Definition of app list changed callback
 *
 * @param pContext The user context registered with MSPIN_CreateInstance
 */
typedef void (*MSPIN_OnAppListChanged)(void* pContext);

/**
 * @func MSPIN_OnAppIconResponse
 *
 * Definition of app icon callback
 *
 * @param pContext The user context registered with MSPIN_CreateInstance
 * @param appIcon The app icon struct
 */
typedef void (*MSPIN_OnAppIconResponse)(void* pContext, MSPIN_AppIconResponse_t* appIcon);

/**
 * @func MSPIN_OnAppStartedResponse
 *
 * Definition of app started response callback
 *
 * @param pContext The user context registered with MSPIN_CreateInstance
 * @param identifierURL The app identification
 * @param appStartStatus The state of the started app
 */
typedef void (*MSPIN_OnAppStartedResponse)(void* pContext, const char* identifierURL, MSPIN_APP_START_STATE appStartStatus);

/**
 * @func MSPIN_OnFrameUpdateRaw
 *
 * Definition of FrameUpdateRaw callback
 *
 * @param pContext The user context registered with MSPIN_CreateInstance
 * @param currentCount The number of the rectangle
 * @param x0 starting point coordinate
 * @param y0 starting point coordinate
 * @param width width of rectangle
 * @param height height
 * @param buffer pointer to the buffer containing the pixel data
 * @param bufferSize Number of bytes provided with the buffer
 * @param scaleHorizontal scale value in horizontal direction
 * @param scaleVertical scale value in vertical direction
 * @param format format of pixel in data
 * @param endianess byte order of pixel data (ePIXELENDIANESS_LittleEndian = 2)
 * @param compression type of compression used to pack the data
 */
typedef void (*MSPIN_OnFrameUpdateRaw)(void* pContext, U8 currentCount, U16 x0, U16 y0, U16 width, U16 height,
        U8* buffer, U32 bufferSize, U8 scaleHorizontal, U8 scaleVertical,
        MSPIN_PIXEL_FORMAT format, U8 endianess, MSPIN_FRAME_COMPRESSION compression);

/**
 * @func MSPIN_Init
 *
 * The function initializes the mySPIN TargetAdapter.
 * Has to be called before any other functions are called
 *
 * @return error code.
 */
MSPIN_ERROR MSPIN_Init(void);

/**
 * @func MSPIN_Shutdown
 *
 * will shutdown the mySPIN TargetAdapter.
 */
void MSPIN_Shutdown(void);

/**
 * @func MSPIN_CreateInstance
 *
 * create an instance
 *
 * @param context The caller might pass its context here, it will be passed back in the callbacks.
 * @return instanceHandle. This handle has to be used in most other functions. On error NULL is returned
 */
MSPIN_Instance_t* MSPIN_CreateInstance(void* context);

/**
 * @func MSPIN_DeleteInstance
 *
 * delete the instance
 *
 * @param[in,out] ppInstanceHandle A pointer to the instance handle. The instance handle
 *      (*ppInstanceHandle) will be set to NULL at the end of the operation.
 */
void MSPIN_DeleteInstance(MSPIN_Instance_t **ppInstanceHandle);


// Setting callbacks and properties (before connecting):

/**
 * @func MSPIN_SetCapabilities
 *
 * defines which capabilities (features) mySPIN can use on the headunit
 *
 * @param instanceHandle The instance handle
 * @param caps bitmask: use enum values from MSPIN_CLIENTCAPABILITIES with bitwise or
 */
void MSPIN_SetCapabilities(MSPIN_Instance_t* instanceHandle, U32 caps);

/**
 * @func MSPIN_SetPhoneCallStatusCallback
 *
 * registers a callback for changes in the phone call status.
 *
 * @param instanceHandle The instance handle
 * @param phoneCallStatusCB Callback will be called when phone call starts or ends
 */
void MSPIN_SetPhoneCallStatusCallback(MSPIN_Instance_t* instanceHandle, void(*phoneCallStatusCB)(void* context, bool isActive));

/**
 * @func MSPIN_SetAppTransitionStatusCallback
 *
 * registers a callback for changes in the app transition status.
 *
 * @param instanceHandle The instance handle
 * @param appTransitionStatusCB Callback will be called when a transition of the app starts or ends
 */
void MSPIN_SetAppTransitionStatusCallback(MSPIN_Instance_t* instanceHandle, void(*appTransitionStatusCB)(void* context,
        bool isActive));

/**
 * @func MSPIN_SetAppInactiveCallback
 *
 * registers a callback for changes in the app inactivity status.
 *
 * @param instanceHandle The instance handle
 * @param appInactiveCB Callback will be called when the app becomes active or inactive
 */
void MSPIN_SetAppInactiveCallback(MSPIN_Instance_t* instanceHandle, void(*appInactiveCB)(void* context,
        bool isInactive));

/**
 * @func MSPIN_SetBlockStatusCallback
 *
 * registers a callback for changes in the block status.
 *
 * @param instanceHandle The instance handle
 * @param blockStatusCB Callback will be called when the app on the phone gets blocked or unblocked
 */
void MSPIN_SetBlockStatusCallback(MSPIN_Instance_t* instanceHandle, void(*blockStatusCB)(void* context, bool isBlocked));

/**
 * @func MSPIN_SetPTTAvailableCallback
 *
 * registers a callback for changes of PTT ('push to talk') available status.
 *
 * @param instanceHandle The instance handle
 * @param pttAvailableCB Callback will be called when a mySPIN application on phone gets started which supports/does not support PTT button
 */
void MSPIN_SetPTTAvailableCallback(MSPIN_Instance_t* instanceHandle, void(*pttAvailableCB)(void* context, bool pttAvailable));

/**
 * @func MSPIN_SetLauncherStateCallback
 *
 * registers a callback for changes in the launcher status
 *
 * @param instanceHandle The instance handle
 * @param launcherStateChangeCB Callback will be called whenever the launcher state changes
 * @return MSPIN_SUCCESS when successful, \
 *         MSPIN_ERROR_NULL_HANDLE when the instance handle is NULL, \
 *         MSPIN_ERROR_NOT_SUPPORTED when not using IVI Core 1.2 or newer
 */
MSPIN_ERROR MSPIN_SetLauncherStateCallback(MSPIN_Instance_t* instanceHandle, MSPIN_OnLauncherStateChange launcherStateChangeCB);

/**
 * @func MSPIN_SetHomeButtonUsableCallback
 *
 * registers a callback for notifications when the home button becomes usable/not usable. This callback is based on the
 * launcherState information.
 *
 * @param instanceHandle The instance handle
 * @param homeButtonUseableCB Callback will be called whenever the state of the usability of the home button changes
 * @return MSPIN_SUCCESS when successful, \
 *         MSPIN_ERROR_NULL_HANDLE when the instance handle is NULL, \
 *         MSPIN_ERROR_NOT_SUPPORTED when not using IVI Core 1.2 or newer
 */
MSPIN_ERROR MSPIN_SetHomeButtonUsableCallback(MSPIN_Instance_t* instanceHandle, MSPIN_OnHomeButtonUsabilityChange homeButtonUsableChangeCB);

/**
 * @func MSPIN_SetInitiatePhoneCallCallback
 *
 * registers a callback for initiating a phone call.
 *
 * @param instanceHandle The instance handle
 * @param phoneCallStatusCB Callback will be called to initiate a phone call
 */
void MSPIN_SetInitiatePhoneCallCallback(MSPIN_Instance_t* instanceHandle,
        void(*initiatePhoneCallCB)(void* context, char* numberString, char* displayString));

/**
 * @func MSPIN_SetNavigateToCallback
 *
 * registers a callback for start navigation to the provided position.
 *
 * @param instanceHandle The instance handle
 * @param navigateToCB Callback will be called to start navigation
 */
void MSPIN_SetNavigateToCallback(MSPIN_Instance_t* instanceHandle,
        void (*navigateToCB) (void* context, char* displayString, MSPIN_NAVIGATETOTYPE type,
                double longitude, double latitude, MSPIN_LocationStrings_t* locationDesciption));
/**
 * @func MSPIN_SetVehicleDataRequestCallback
 *
 * registers a callback for vehicle data request (to register updates).
 * With this callback the running mySPIN application can tell the IVI that it is or is not any more interested into the
 * listed vehicle data. Only when request is set to true, the listed vehicle data should be transferred to the phone.
 *
 * @param instanceHandle The instance handle
 * @param vehicleDataRequestCB Callback will be called to save the keys (vehicle data) which
 *        should be provided to the core/smartphone
 */
void MSPIN_SetVehicleDataRequestCallback(MSPIN_Instance_t* instanceHandle,
        void (*vehicleDataRequestCB) (void* context, bool request, U8 length, U32* keyList));

/**
 * @func MSPIN_SetCustomDataStringCallback
 *
 * registers a callback to allow the phone to provide a message.
 *
 * @param instanceHandle The instance handle
 * @param customStringCB Callback will be called to provide a string message
 */
void MSPIN_SetCustomDataStringCallback(MSPIN_Instance_t* instanceHandle,
        void(*customStringCB)(void* context, MSPIN_PHONEMSGSTRTYPE type, MSPIN_STRINGENCODING encoding, char* data));

/**
 * @func MSPIN_SetCustomIntCallback
 *
 * registers a callback to allow the phone to provide integer data.
 *
 * @param instanceHandle The instance handle
 * @param customStringCB Callback will be called to provide integer data
 */
void MSPIN_SetCustomIntCallback(MSPIN_Instance_t* instanceHandle,
        void(*customIntCB)(void* context, MSPIN_PHONEMSGINTTYPE type, U16 length, U8* data));

/**
 * @func MSPIN_SetAppListChangedCallback
 *
 * registers a callback to allow the phone to notify about a (changed) app list.
 *
 * @param instanceHandle The instance handle
 * @param appListChangedCB Callback will be called to notify about app list change
 * @return MSPIN_SUCCESS when successful, \
 *         MSPIN_ERROR_NULL_HANDLE when the instance handle is NULL, \
 *         MSPIN_ERROR_NOT_SUPPORTED when not using IVI Core 1.2 or newer
 */
MSPIN_ERROR MSPIN_SetAppListChangedCallback(MSPIN_Instance_t* instanceHandle, MSPIN_OnAppListChanged appListChangedCB);

/**
 * @func MSPIN_SetAppIconResponseCallback
 *
 * registers a callback to allow the phone to provide an app icon.
 *
 * @param instanceHandle The instance handle
 * @param appIconResponseCB Callback will be called to provide app icon
 * @return MSPIN_SUCCESS when successful, \
 *         MSPIN_ERROR_NULL_HANDLE when the instance handle is NULL, \
 *         MSPIN_ERROR_NOT_SUPPORTED when not using IVI Core 1.2 or newer
 */
MSPIN_ERROR MSPIN_SetAppIconResponseCallback(MSPIN_Instance_t* instanceHandle, MSPIN_OnAppIconResponse appIconResponseCB);

/**
 * @func MSPIN_SetAppStartedResponseCallback
 *
 * registers a callback to allow the phone to provide status of started app.
 *
 * @param instanceHandle The instance handle
 * @param appStartedResponseCB Callback will be called to provide status of started app
 * @return MSPIN_SUCCESS when successful, \
 *         MSPIN_ERROR_NULL_HANDLE when the instance handle is NULL, \
 *         MSPIN_ERROR_NOT_SUPPORTED when not using IVI Core 1.2 or newer
 */
MSPIN_ERROR MSPIN_SetAppStartedResponseCallback(MSPIN_Instance_t* instanceHandle,
        MSPIN_OnAppStartedResponse appStartedResponseCB);

/**
 * @func MSPIN_SetVoiceSupport
 *
 * enables/disables voice support with the specified functionality. Enabling this feature depends on the capabilities
 * of the IVI system. If enabled, make sure that the callback is also registered with
 * MSPIN_SetVoiceSessionRequestCallback.
 * Call this function during initializing of a mySPIN session.
 *
 * @param instanceHandle The instance handle
 * @param supportType The voice session support type
 * @param constraint The constraints (none or microphone and speaker together only. See MSPIN_VOICESESSION_CONSTRAINT)
 * @return MSPIN_SUCCESS when successful. Otherwise an error.
 */
MSPIN_ERROR MSPIN_SetVoiceSupport(MSPIN_Instance_t* instanceHandle, MSPIN_VOICESESSION_SUPPORT supportType,
        MSPIN_VOICESESSION_CONSTRAINT constraint);

/**
 * @func MSPIN_SetVoiceSessionRequestCallback
 *
 * registers a callback to allow the phone to request a voice session.
 * Voice session requests will only be supported since mySPIN IVI Core 1.1
 *
 * @param instanceHandle The instance handle
 * @param voiceSessionRequestCB Callback will be called to control the voice session
 * @return MSPIN_Success when successful. In case this feature isn't supported, the result code is
 *         MSPIN_ERROR_NOT_IMPLEMENTED
 */
MSPIN_ERROR MSPIN_SetVoiceSessionRequestCallback(MSPIN_Instance_t* instanceHandle,
        void(*voiceSessionRequestCB)(void* context, MSPIN_VOICESESSION_REQUESTTYPE requestType));

/**
 * @func MSPIN_UpdateVoiceSessionStatus
 *
 * Updates the status of voice session. Use this only when voice session support is enabled with MSPIN_SetVoiceSupport
 * and after the mySPIN session was successfully established..
 *   Per default the status should be unavailable.
 *   Set status to available when you are able to receive a voice session (e.g. no incoming call)
 *   Answer a voiceSessionRequestCB of request type startXXX with MSPIN_VOICESESSION_STATUS_START_REQUESTED when you
 *   are ready
 *   Answer a voiceSessionRequestCB of request type startXXX with MSPIN_VOICESESSION_STATUS_START_REQUESTED
 *   When the voice session is started on IVI set voice session status to eVOICESESSIONSTATUS_Active to indicate that
 *   the voice session is active
 *
 * @param instanceHandle The instance handle
 * @param status The voice session status to be set
 * @param constraint The voice session constraint. In case the voice session is not available, here a reason should be
 *                   given why the voice session is not available. If available, use MSPIN_VOICESESSION_STATUS_CONSTRAINT_NONE.
 * @return MSPIN_Success when successful. Errors are:
 *          - MSPIN_ERROR_INVALID_PARAMETER: instanceHandle is NULL
 *          - MSPIN_ERROR_NOT_IMPLEMENTED: voice session isn't supported
 *          - MSPIN_ERROR_GENERAL: voice session isn't enabled with MSPIN_SetVoiceSupport
 */
MSPIN_ERROR MSPIN_UpdateVoiceSessionStatus(MSPIN_Instance_t* instanceHandle, MSPIN_VOICESESSION_STATUS status,
        MSPIN_VOICESESSION_STATUS_CONSTRAINT constraint);

/**
 * @func MSPIN_SetAudioRequestCallback
 *
 * registers a callback to allow the phone to send audio request.
 * Audio requests will only be supported since mySPIN IVI Core 1.1
 *
 * @param instanceHandle The instance handle
 * @param audioRequestCB Callback will be called to request or release audio
 * @return MSPIN_Success when successful. In case this feature isn't supported, the result code is
 *         MSPIN_ERROR_NOT_IMPLEMENTED
 */
MSPIN_ERROR MSPIN_SetAudioRequestCallback(MSPIN_Instance_t* instanceHandle,
        void (*audioRequestCB)(void* context, MSPIN_AUDIOCONTROL command, U32 requestID, MSPIN_AUDIOTYPE type));

/**
 * @func MSPIN_SetErrorCallback
 *
 * registers a callback to report errors.
 *
 * @param instanceHandle The instance handle
 * @param errorCB Callback will be called when an error occurs. This instance cannot be used any longer and should be deleted.
 */
void MSPIN_SetErrorCallback(MSPIN_Instance_t* instanceHandle, void(*errorCB)(void* context, MSPIN_ERROR error));

/**
 * @func MSPIN_SetLayerManagerInitializedCallback
 *
 * registers a callback to get a notification when the LayerManager is successfully initialized
 *
 * @param instanceHandle The instance handle
 * @param LMInitializedCB Callback to be called when LayerManager is successfully initialized (=ready to use)
 */
void MSPIN_SetLayerManagerInitializedCallback(MSPIN_Instance_t* instanceHandle, void(*LMInitializedCB)(void* context));

/**
 * @func MSPIN_SetLayerManagerDeinitializedCallback
 *
 * registers a callback to get a notification before the LayerManager gets deinitialized
 *
 * @param instanceHandle The instance handle
 * @param LMDeinitializedCB Callback to be called when LayerManager gets deinitialized (=cannot be used anymore)
 */
void MSPIN_SetLayerManagerDeinitializedCallback(MSPIN_Instance_t* instanceHandle, void(*LMInitializedCB)(void* context));

/**
 * @func MSPIN_SetFirstFrameRenderedCallback
 *
 * registers a callback to get a notification when the first frame is rendered for example after connection
 * establishment or resume. The callback FirstFrameRenderedCB will be called for the first frame received
 * after calls to the following functions:
 * - MSPIN_ConnectAoap() (Android case)
 * - MSPIN_EAPSessionStart() (iPhone case)
 * - MSPIN_ResumeConnection()
 *
 * @param instanceHandle The instance handle
 * @param FirstFrameRenderedCB Callback to be called when the first frame is rendered successfully. The returned context is set
 *                             within MSPIN_CreateInstance().
 */
void MSPIN_SetFirstFrameRenderedCallback(MSPIN_Instance_t* instanceHandle, void(*FirstFrameRenderedCB)(void* context));

/**
 * @func MSPIN_SetFrameTransmissionStatusCallback
 *
 * registers a callback to get a notification when the status of the frame transmission changes (starts or stops). Default state after connection setup
 * is stopped. This means that the first notification is typically a start notification.
 *
 * This notification is especially sent when
 * a) a connection is established and frame transmission starts
 * b) a connection is suspended/resumed
 * c) an app transition is started/finished and frame transmission state changed
 *
 * Note: The cases a) and b) are also handled via MSPIN_SetFirstFrameRenderedCallback.
 *
 * @param instanceHandle The instance handle
 * @param FrameTransmisionStatusCB Callback will be issued when the frame transmission state changes. The returned context is set
 *                                 within MSPIN_CreateInstance(). The flag start states if the frame transmission starts or stops.
 */
void MSPIN_SetFrameTransmissionStatusCallback(MSPIN_Instance_t* instanceHandle, void(*FrameTransmisionStatusCB)(void* context, bool start));

/**
 * @func MSPIN_SetFrameUpdateRawCallback
 *
 * register frame update callback for delivery of compressed ('raw') frame data
 * Setting this callback will automatically switch off uncompressing by the core (for the next connection).
 *
 * @param instanceHandle The instance handle
 * @param frameUpdateRawCB This function will be called on each new rectangle to be updated.
 */
void MSPIN_SetFrameUpdateRawCallback(MSPIN_Instance_t* instanceHandle, MSPIN_OnFrameUpdateRaw frameUpdateRawCB);

/**
 * @func MSPIN_SetWaylandTouch
 *
 * can be used to enable or disable TouchEvents from Wayland. By default TouchEvents from Wayland are switched on.
 *
 * @param instanceHandle The instance handle
 * @param enable  True to switch on, false to switch off
 */
void MSPIN_SetWaylandTouch(MSPIN_Instance_t* instanceHandle, bool enable);

/**
 * @func MSPIN_EnableTouchFiltering
 *
 * enables touch filtering for touch move events.
 * Move events will not be forwarded when enabled if
 * a) the position did not change and
 * b) the last move event is within the last 25ms (see define MSPIN_LM_MINIMUM_TOUCH_INTERVAL_IN_MS)
 * The first move event is immediately sent when the position is different to the touch down event
 *
 * @param instanceHandle The instance handle
 * @param enable When set to TRUE touch filtering gets enabled. If FALSE touch filtering gets disabled. Per
 * default touch filtering is disabled.
 *
 * @return error code.
 */
MSPIN_ERROR MSPIN_EnableTouchFiltering(MSPIN_Instance_t* instanceHandle,  bool enable);

/**
* @func MSPIN_EnablePointerFiltering
*
* enables pointer filtering for pointer move events.
* Move events will not be forwarded when enabled if
* a) the position did not change and
* b) the last move event is within the last 25ms (see define MSPIN_LM_MINIMUM_POINTER_INTERVAL_IN_MS)
* The first move event is immediately sent when the position is different to the pointer button event
*
* @param instanceHandle The instance handle
* @param enable When set to TRUE touch filtering gets enabled. If FALSE pointer filtering gets disabled. Per
* default pointer filtering is disabled.
*
 * @return error code.
*/
MSPIN_ERROR MSPIN_EnablePointerFiltering(MSPIN_Instance_t* instanceHandle,  bool enable);

/**
 * @func MSPIN_SetFrameProperties
 *
 * is used to set the frame properties.
 *
 * @param instanceHandle The instance handle
 * @param frameProps Struct with the frame properties.
 *        Note: The compression is normally chosen automatically since IVI Core 1.1.x by launcher/SDK on phone from
 *        the available ones. In some cases it might be necessary to override compression. In this case set
 *        'preferredCompression' to something different than 'MSPIN_FRAME_COMPRESSION_NONE'. Please align with
 *        Bosch SoftTec when using this.
 */
void MSPIN_SetFrameProperties(MSPIN_Instance_t* instanceHandle, MSPIN_FrameProperties_t *frameProps);

/**
 * @func MSPIN_SetAccessoryPropertiesAoap
 *
 * is used to provide the necessary strings for AOAP to identify the matching app on the Android phone.
 *
 * @param instanceHandle The instance handle
 * @param accProps Struct with the accessory properties (the accessory is the headunit)
 */
void MSPIN_SetAccessoryPropertiesAoap(MSPIN_Instance_t* instanceHandle, MSPIN_AccessoryPropertiesAoap_t *accProps);

/**
 * @func MSPIN_SetISO639LanguageCode
 *
 * is used to provide the HMI language to the phone. (Will be used e.g. for app names in app info list.)
 *
 * @param instanceHandle The instance handle - can be NULL if called before instance is created.
 * @param languageCode Language code in ISO 639-2 format (e.g. 'eng').
 */
void MSPIN_SetISO639LanguageCode(MSPIN_Instance_t* instanceHandle, U8 languageCode[3]);

/**
 * @func MSPIN_SetInitializationTimeout
 *
 * change the initialization timeout for the mySPIN Core to setup the protocol with the phone.
 * Default is 10 seconds.
 *
 * @param instanceHandle The instance handle
 * @param seconds number of seconds for the initialization timeout. Use 0 to disable/wait infinitely.
 */
void MSPIN_SetInitializationTimeout(MSPIN_Instance_t* instanceHandle, U8 seconds);

/**
 * @func MSPIN_MeasurePerformanceAOAP
 *
 * Enables performance measurements for AOAP read and write operations. This call is only intended
 * for debugging purposes.
 *
 * ATTENTION: Do not use in production code!!!
 */
void MSPIN_MeasurePerformanceAOAP(void);

/**
 * @func MSPIN_MeasureFrameLatency
 *
 * Enables frame latency measurements on info log level. This call is only intended for performance
 * optimizations.
 *
 * ATTENTION: Do not use in production code!!!
 */
void MSPIN_MeasureFrameLatency(void);

/**
 * @func MSPIN_CountFramesPerSecond
 *
 * Enables counting of frames transmitted and rendered from smartphone on the headunit. Print outs
 * will indicate the frames per second. At least warning logging level must be enabled. This call
 * is only intended for debugging purposes. Attention: Do not use in production code!
 */
void MSPIN_CountFramesPerSecond(void);


//Connect:

/**
 * @func MSPIN_ConnectAoap
 *
 * tries to establish an AOAP connection to that phone and starts the mySPIN protocol. This function is synchronous,
 * i.e. it blocks until the phone is successfully connected or an error is detected. This might take at maximum a few seconds.
 *
 * Caution: Do not connect twice with the same instance - always connect to a newly created instance!
 *
 * @param instanceHandle The instance handle
 * @param vendorId Vendor ID
 * @param productId Product ID
 * @param serial Serial number
 * @param audioSupport Enable/disable audio support for this device. Disabling audio support might be
 *           necessary for some Android devices which fail to switch to accessory mode when audio is
 *           enabled (like the Samsung S3 with Android 4.1.2).
 *           Parameter by reference; value will be changed to false, if AOAP can't switch on audio.
 *
 * @return error code. MSPIN_SUCCESS on success, \
 *      MSPIN_ERROR_CONNECTION_START if failed to connect to the given phone (AOAP), \
 *      MSPIN_ERROR_CORE if mySPIN Core initialization failed, \
 *      MSPIN_ERROR_LAYERMANAGER if layer manager initialization failed, \
 *      MSPIN_ERROR_GENERAL on general error
 * @return audioSupport see description at param audioSupport above
 */
MSPIN_ERROR MSPIN_ConnectAoap(MSPIN_Instance_t* instanceHandle, U32 vendorId, U32 productId,
        const char* serial, bool* audioSupport);

/**
 * @func MSPIN_DisconnectAoap
 *
 * Disconnects AOAP device identified via mySPIN instance handle
 * Note, that the phone will stay in accessory mode. So, probably we cannot re-connect to app on phone.
 *
 * @param[in] instanceHandle The mySPIN instance handle for identification of the connection
 */
void MSPIN_DisconnectAoap(MSPIN_Instance_t *pInstanceHandle);

/**
 * @func MSPIN_ConnectiAP2
 *
 * Sets up the connection parameters for iAP2 connection but does not establish the iAP2
 * connection. Establishment of the iAP2 connection is not in the responsibility of mySPIN target adapter.
 *
 * This function is synchronous but there is no time consuming task to do inside this function.
 *
 * @param[in,out] pInstanceHandle The instance handle
 * @param vendorId USB vendor ID which should be the Apple vendor identifier.
 * @param productId USB product ID which should be one of Apple's iOS device product identifiers.
 * @param pSerial USB serial number of the iOS device
 * @param pWriteDevice The device name for writing (e.g. '/dev/ffs/ep3' when using GadgetFS/FunctionFS)
 * @param pReadDevice The device name for reading (e.g. '/dev/ffs/ep4' when using GadgetFS/FunctionFS)
 * @param hostMode TRUE when EA Native Transport mode is used. This is default. When EAP Session
 * shall be used, this must be set to FALSE.
 *
 * @return error code. MSPIN_SUCCESS on success, \
 *      MSPIN_ERROR_GENERAL when memory allocation fails
 */
MSPIN_ERROR MSPIN_ConnectiAP2(MSPIN_Instance_t *pInstanceHandle,
        U32 vendorId,
        U32 productId,
        const char *pSerial,
        const char *pWriteDevice,
        const char *pReadDevice,
        bool hostMode);

/**
 * @func MSPIN_DisconnectiAP2
 *
 * Disconnects iOS device identified via mySPIN instance handle
 *
 * @param[in] instanceHandle The mySPIN instance handle for identification of the connection
 */
void MSPIN_DisconnectiAP2(MSPIN_Instance_t *pInstanceHandle);

/**
 * @func MSPIN_SetEAPReadWriteCallbacks
 *
 * Sets the read/write callbacks for EAP Session in device mode.
 *
 * @param pInstanceHandle[in,out] pInstanceHandle The instance handle
 * @param eapSend_CB The send data callback
 * @param eapReceive_CB The receive data callback
 * @param pEapContext The context returned in eapSend_CB and eapReceive_CB
 * @return error code. MSPIN_SUCCESS on success, \
 *      MSPIN_ERROR_GENERAL if instanceHandle or connectionHanle of instanceHanle is NULL
 */
MSPIN_ERROR MSPIN_SetEAPReadWriteCallbacks(MSPIN_Instance_t *pInstanceHandle,
        MSPIN_EAPSend eapSend_CB, MSPIN_EAPRead eapReceive_CB, void *pEapContext);

/**
 * @func MSPIN_EAPSessionStart
 *
 * Forwards the EAP session start event (start of the accessory application) which will start the
 * mySPIN Core
 *
 * @param[in] instanceHandle The instance handle
 *
 * @return error code. MSPIN_SUCCESS on success, \
 *      MSPIN_ERROR_GENERAL if instanceHandle is NULL or if the reader thread couldn't be initialized
 */
MSPIN_ERROR MSPIN_EAPSessionStart(MSPIN_Instance_t *pInstanceHandle);

/**
 * @func MSPIN_EAPSessionStop
 *
 * Forwards the EAP session start event (stop of the accessory application). This will stop the
 * mySPIN Core if not yet stopped
 *
 * @param[in] instanceHandle The instance handle
 */
void MSPIN_EAPSessionStop(MSPIN_Instance_t *pInstanceHandle);

/**
 * @func MSPIN_ConnectTCPServer
 *
 * Tries to establish a TCP/IP connection to the specified server and port.
 *
 * @param pInstanceHandle The instance handle
 * @param hostname The hostname. Typically this is the IP address
 * @param port The port number the mySPIN service is listening on. Could be received from the mySPIN UDP message
 * @return error code. MSPIN_SUCCESS on success, \
 *                     different error codes are possible
 */
MSPIN_ERROR MSPIN_ConnectTCPServer(MSPIN_Instance_t *pInstanceHandle, const U8 *hostname, unsigned int port);

/**
 * @func MSPIN_DisconnectTCPServer
 *
 * To disconnect a TCP/IP connection to a server.
 *
 * @param pInstanceHandle The instance handle
 * @return error code. MSPIN_SUCCESS on success, \
 *                     different error codes are possible
 */
MSPIN_ERROR MSPIN_DisconnectTCPServer(MSPIN_Instance_t *pInstanceHandle);

/**
 * @func MSPIN_StartUDPListener
 *
 * Starts a UDP broadcast listener. When a UDP broadcast packet is received, the registered callback \
 * mesgReceivedCB is called with the given context. In order to stop UDP broadcast listener call MSPIN_StopUDPListener.
 *
 * @param[in] port The port number to be used by the UDP listener
 * @param[in] mesgReceivedCB The callback to be called when a valid mySPIN UDP broadcast message is received. The
 *                           callback contains the context pUDPListenerContext, the hostname and port number extracted
 *                           from the UDP broadcast message if it's a valid mySPIN broadcast message.
 * @param[in] pUDPListenerContext The context to be returned with calls of mesgReceivedCB
 * @return error code. MSPIN_SUCCESS on success, \
 *                     MSPIN_ERROR_NOT_READY if another listener is already registered. Issue MSPIN_StopUDPListener() \
 *                                           first to deregister the present callback \
 *                     MSPIN_ERROR_GENERAL in case the UDP listener could not be started \
 *                     MSPIN_ERROR_NOT_SUPPORTED if the feature is not supported \
 *                     MSPIN_ERROR_MEM_ALLOC if memory allocation fails
 */
MSPIN_ERROR MSPIN_StartUDPListener(S32 port, MSPIN_UDPMessageReceived mesgReceivedCB, void *pUDPListenerContext);

/**
 * @func MSPIN_StopUDPListener
 *
 * Stops the UDP broadcast listener started with MSPIN_StartUDPListener
 *
 * @return error code. MSPIN_SUCCESS on success, \
 *                     MSPIN_ERROR_NOT_SUPPORTED if the feature is not supported
 */
MSPIN_ERROR MSPIN_StopUDPListener(void);

/**
 * @brief MSPIN_StartTCPListener
 *
 * Starts listening on the specified TCP port. This TCP port may not be blocked by any firewall otherwise connections
 * could not be accepted.
 * The same TCP port should be used in MSPIN_StartUDPBroadcasting to notify any interested mySPIN launcher to which IP
 * and port it could connect.
 *
 * @param port The TCP port on which socket connections will be accepted
 * @param acceptCB The callback notifies the user that a new connection got accepted. The user can either close it
 *                 directly by returning false or close it later. If true is returned, the user can open a mySPIN
 *                 session and close it (which closes also the socket) or close the socket later manually with
 *                 MSPIN_CloseConnection.
 *                 Attention: This function should return immediately. It's not allowed to call from this callback
 *                 MSPIN_ConnectTCPClient.
 * @param closedCB The callback notifies the user that a connection got closed. The reason and the IP v4 address is \
 *                 given within the callback. The connection ID might be invalid (-1) when the connection has not \
 *                 been accepted so far.
 * @param callbackContext A user context passed back in the callbacks registered with this function
 * @return error code. MSPIN_SUCCESS on success, \
 *                     MSPIN_ERROR_NOT_SUPPORTED in case TCP/IP connections are disabled by compile flag, \
 *                     MSPIN_ERROR_GENERAL in case a failure happens during setup of the listener, \
 *                     MSPIN_ERROR_NOT_FOUND in case the TLS listener is running and not the TCP/IP one
 */
MSPIN_ERROR MSPIN_StartTCPListener(S32 port, MSPIN_OnAcceptIncomingConnection acceptCB,
        MSPIN_OnConnectionClosed closedCB, void* callbackContext);

/**
 * @brief MSPIN_StopTCPListener
 *
 * Stops the TCP listener. If after the call new incoming connections arrive, these requests will be refused. \
 * Active connections are not affected. The closedCB registered with MSPIN_StartTCPListener will be also deregistered \
 * and therefore no longer be called.
 *
 * @return error code. MSPIN_SUCCESS on success, \
 *                     MSPIN_ERROR_NOT_SUPPORTED in case TCP/IP connections are disabled by compile flag
 */
MSPIN_ERROR MSPIN_StopTCPListener(void);

/**
 * @brief MSPIN_StartTLSListener
 *
 * Starts listening on the specified port. This TCP port may not be blocked by any firewall otherwise connections
 * could not be accepted.
 * The same TCP port should be used in MSPIN_StartUDPBroadcasting to notify any interested mySPIN launcher to which IP
 * and port it could connect.
 *
 * @param port The TCP port on which socket connections will be accepted
 * @param tlsConfig Required configuration parameters of TLS (server's certificate and private key file, CA file, \
 *                  options)
 * @param acceptCB The callback notifies the user that a new connection got accepted. The user can either close it
 *                 directly by returning false or close it later. If true is returned, the user can open a mySPIN
 *                 session and close it (which closes also the socket) or close the socket later manually with
 *                 MSPIN_CloseConnection.
 *                 Attention: This function should return immediately. It's not allowed to call from this callback
 *                 MSPIN_ConnectTCPClient.
 * @param closedCB The callback notifies the user that a connection got closed. The reason and the IP v4 address is \
 *                 given within the callback. The connection ID might be invalid (-1) when the connection has not \
 *                 been accepted so far.
 * @param callbackContext A user context passed back in the callbacks registered with this function
 * @return error code. MSPIN_SUCCESS on success, \
 *                     MSPIN_ERROR_NOT_SUPPORTED in case TCP/IP connections are disabled by compile flag, \
 *                     MSPIN_ERROR_GENERAL in case a failure happens during setup of the listener, \
 *                     MSPIN_ERROR_NOT_FOUND in case the TCP/IP listener is running and not the TLS one
 */
MSPIN_ERROR MSPIN_StartTLSListener(S32 port, MSPIN_TLS_CONFIGURATION_t* tlsConfig, MSPIN_OnAcceptIncomingConnection acceptCB,
        MSPIN_OnConnectionClosed closedCB, void* callbackContext);

/**
 * @brief MSPIN_StopTCPListener
 *
 * Stops the TLS listener. If after the call new incoming connections arrive, these requests will be refused.
 * Active connections are not affected.
 *
 * @return MSPIN_SUCCESS on success, \
 *         MSPIN_ERROR_NOT_SUPPORTED in case TCP/IP connections are disabled by compile flag
 */
MSPIN_ERROR MSPIN_StopTLSListener(void);

/**
 * @brief MSPIN_CloseConnection
 *
 * Closes the socket connection identified by the connectionID (returned in MSPIN_OnAcceptIncomingConnection).
 *
 * @param connectionID The connection ID identifying the TCP/IP connection to be terminated.
 * @return error code. MSPIN_SUCCESS on success, \
 *                     MSPIN_ERROR_NOT_FOUND when the connectionID could not be found for example because it got already closed, \
 *                     MSPIN_ERROR_INVALID_PARAMETER in case the socketFD is not valid.
 */
MSPIN_ERROR MSPIN_CloseConnection(S32 connectionID);

/**
 * @brief MSPIN_GetMacAddress
 *
 * Get the MAC address of the specified LAN connection using arp cache. Can be used only on active connections.
 * The output string for the MAC address must be at minimum 18 bytes long.
 *
 * @param connectionID The connection ID identifying the TCP/IP connection from which the MAC address should be retrieved
 * @param macAddress The string holding the retrieved MAC address e.g. 7E:78:61:A1:56:BA. Letters are uppercase. The macAddress string must have at least 18 bytes size.
 * @return error code. MSPIN_SUCCESS on success, \
 *                     MSPIN_ERROR_NOT_FOUND when the connectionID could not be found for example because it got already closed, \
 *                     MSPIN_ERROR_NULL_HANDLE in case any input parameter is NULL, \
 *                     MSPIN_ERROR_GENERAL retrieving MAC address failed for some other reason.
 */
MSPIN_ERROR MSPIN_GetMacAddress(S32 connectionID, U8* macAddress);

/**
 * @brief MSPIN_StartUDPBroadcasting
 *
 * Starts UDP broadcasting on the specified udpPort.
 * The broadcast address will be taken from the specified interface.
 * The message to be sent will be automatically generated from the message parameters.
 *
 * @param udpPort The UDP port used for UDP broadcasting
 * @param interface The interface on which the UDP broadcast will be executed and on which incoming TCP connections
 *                  will be handled
 * @param messageParameters The UDP message parameters including the four identification strings and the TCP port. The
 *                          TCP port must be same as in MSPIN_StartTCPListener
 * @param timeout The timeout in milliseconds which specifies the interval in which UDP packets will be transmitted.
 * @param onUDPBroadcastEnd A callback which notifies the user when the UDP broadcasting ends. Only triggered when
 *                          function MSPIN_StartUDPBroadcasting returns with success.
 * @param onUDPBroadcastEndContext A user context which will be passed back in MSPIN_OnUDPBroadcastEnd.
 * @return error code. MSPIN_SUCCESS on success, \
 *                     MSPIN_ERROR_INVALID_PARAMETER in case the message parameters contain something invalid, \
 *                     MSPIN_ERROR_NOT_SUPPORTED in case TCP/IP connections are disabled by compile flag, \
 *                     MSPIN_ERROR_GENERAL in case a failure happens during setup of the UDP broadcasting, \
 *                     MSPIN_ERROR_NOT_READY in case the broadcast address could not be determined e.g.
 *                     because network is not up and running, \
 *                     MSPIN_ERROR_ALREADY_RUNNING in case there is still an UDP broadcast instance. In this case stop
 *                     it first but make sure not to call it from callback onUDPBroadcastEnd.
 */
MSPIN_ERROR MSPIN_StartUDPBroadcasting(S32 udpPort, const U8* interface, MSPIN_UDP_MESSAGE_PARAMETER_t messageParameters, U32 timeout,
        MSPIN_OnUDPBroadcastEnd onUDPBroadcastEnd, void* onUDPBroadcastEndContext);

/**
 * @brief MSPIN_StopUDPBroadcasting
 *
 * Stops the UDP broadcasting. Use this also if a UDP broadcasting is already running and you want to change any
 * parameters.
 *
 * Attention: Do not call it directly from callback 'onUDPBroadcastEnd'. Another thread context is required!
 *
 * @return error code. MSPIN_SUCCESS on success, \
 *                     MSPIN_ERROR_NOT_SUPPORTED in case TCP/IP connections are disabled by compile flag, \
 *                     MSPIN_ERROR_NOT_ALLOWED in case stop is called from the UDB broadcasting thread for example
 *                     when issued from the callback 'onUDPBroadcastEnd', \
 */
MSPIN_ERROR MSPIN_StopUDPBroadcasting(void);

/**
 * @brief MSPIN_StartUDPUnicasting
 *
 * Starts UDP unicasting on the specified udpPort and a destination IP address and only to this destination
 * IP address. The message to be sent will be automatically generated from the message parameters.
 *
 * @param udpPort The UDP port used for UDP broadcasting
 * @param interface The interface on which the UDP unicast will be executed and on which incoming TCP connections
 *                  will be handled
 * @param destIPAddr The IP address of the destination
 * @param messageParameters The UDP message parameters including the four identification strings and the TCP port. The
 *                          TCP port must be same as in MSPIN_StartTCPListener
 * @param timeout The timeout in milliseconds which specifies the interval in which UDP packets will be transmitted.
 * @param onUDPUnicastEnd A user context which will be passed back in MSPIN_OnUDPUnicastEnd.
 * @param onUDPUnicastEndContext
 * @return error code. MSPIN_SUCCESS on success, \
 *                     MSPIN_ERROR_INVALID_PARAMETER in case the message parameters contain something invalid, \
 *                     MSPIN_ERROR_NOT_SUPPORTED in case TCP/IP connections are disabled by compile flag, \
 *                     MSPIN_ERROR_GENERAL in case a failure happens during setup of the UDP broadcasting, \
 *                     MSPIN_ERROR_NOT_READY in case the broadcast address could not be determined e.g.
 *                     because network is not up and running, \
 *                     MSPIN_ERROR_ALREADY_RUNNING in case there is still an UDP broadcast instance. In this case stop
 *                     it first but make sure not to call it from callback onUDPBroadcastEnd, \
 *                     MSPIN_ERROR_INVALID_PARAMETER in case the destIPAddr is not valid.
 */
MSPIN_ERROR MSPIN_StartUDPUnicasting(S32 udpPort, const U8* interface, const U8* destIPAddr, MSPIN_UDP_MESSAGE_PARAMETER_t messageParameters, U32 timeout,
        MSPIN_OnUDPUnicastEnd onUDPUnicastEnd, void* onUDPUnicastEndContext);

/**
 * @brief MSPIN_StopUDPUnicasting
 *
 * Stops the UDP unicasting. Use this also if a UDP unicasting is already running and you want to change any
 * parameters.
 *
 * Attention: Do not call it directly from callback 'onUDPUnicastEnd'. Another thread context is required!
 *
 * @return error code. MSPIN_SUCCESS on success, \
 *                     MSPIN_ERROR_NOT_SUPPORTED in case TCP/IP connections are disabled by compile flag, \
 *                     MSPIN_ERROR_NOT_ALLOWED in case stop is called from the UDB broadcasting thread for example
 *                     when issued from the callback 'onUDPBroadcastEnd', \
 */
MSPIN_ERROR MSPIN_StopUDPUnicasting(void);

/**
 * @brief MSPIN_ConnectTCPClient
 *
 * Establishes in server mode a TCP/IP connection to the client smartphone. The socket connection must be already
 * accepted (compare MSPIN_StartTCPListener and the callback MSPIN_OnAcceptIncomingConnection).
 *
 * @param pInstanceHandle The mySPIN instance handle
 * @param connectionID The connection ID. This must be the one delivered by MSPIN_OnAcceptIncomingConnection.
 * @return error code. MSPIN_SUCCESS on success,\
 *                     MSPIN_ERROR_NOT_SUPPORTED in case TCP/IP connections are disabled by compile flag \
 *                     MSPIN_ERROR_GENERAL when some other error occurs \
 *                     other values
 */
MSPIN_ERROR MSPIN_ConnectTCPClient(MSPIN_Instance_t *pInstanceHandle, S32 connectionID);

/**
 * @brief MSPIN_DisconnectTCPClient
 *
 * Disconnects in server mode a TCP/IP connection to a client smartphone. It also closes the socket connection used for
 * this connection.
 *
 * @param pInstanceHandle The mySPIN instance handle
 * @return error code. MSPIN_SUCCESS on success, \
 *                     MSPIN_ERROR_NOT_SUPPORTED in case TCP/IP connections are disabled by compile flag \
 *                     MSPIN_ERROR_GENERAL in several cases
 */
MSPIN_ERROR MSPIN_DisconnectTCPClient(MSPIN_Instance_t *pInstanceHandle);

/**
 * @brief MSPIN_ConnectTLSClient
 *
 * Establishes in server mode a TLS connection to the client smartphone. The socket connection must be already
 * accepted (compare MSPIN_StartTLSListener and the callback MSPIN_OnAcceptIncomingConnection).
 *
 * @param pInstanceHandle The mySPIN instance handle
 * @param connectionID The connection ID. This must be the one delivered by MSPIN_OnAcceptIncomingConnection.
 * @return error code. MSPIN_SUCCESS on success,\
 *                     MSPIN_ERROR_NOT_SUPPORTED in case TLS and/or TCP/IP connections are disabled by compile flag \
 *                     MSPIN_ERROR_GENERAL when some other error occurs \
 *                     other values
 */
MSPIN_ERROR MSPIN_ConnectTLSClient(MSPIN_Instance_t *pInstanceHandle, S32 connectionID);

/**
 * @brief MSPIN_DisconnectTLSClient
 *
 * Disconnects in server mode a TLS connection to a client smartphone. It also closes the socket connection used for
 * this connection.
 *
 * @param pInstanceHandle The mySPIN instance handle
 * @return error code. MSPIN_SUCCESS on success, \
 *                     MSPIN_ERROR_NOT_SUPPORTED in case TLS and or TCP/IP connections are disabled by compile flag \
 *                     MSPIN_ERROR_GENERAL in several cases
 */
MSPIN_ERROR MSPIN_DisconnectTLSClient(MSPIN_Instance_t *pInstanceHandle);

// Operation (while connected):

/**
 * @func MSPIN_GetAvailableKeys
 *
 * get the available keys from the connected phone. The headunit should provide these keys
 * while operating the phone (keys can be shown as softkeys or hardkeys can be used).
 * In the returned bitmask the following bits/values represent these keys:
 *    1 = Home
 *    2 = Back
 *    4 = Menu
 *    8 = Search
 *
 * @param instanceHandle The instance handle
 * @return bitmask representing the keys
 */
U32 MSPIN_GetAvailableKeys(MSPIN_Instance_t* instanceHandle);

/**
 * @func MSPIN_GetServerCapabilities
 *
 * get server capabilities
 *
 * @param instanceHandle The instance handle
 * @return bitmask representing the server capabilities. Use values from MSPIN_SERVERCAPABILITIES to match bits.
 *         In case the the instanceHandle, the core instance is NULL or using mySPIN IVI Core 1.0.x the result
 *         will be 0.
 */
U32 MSPIN_GetServerCapabilities(MSPIN_Instance_t* instanceHandle);

/**
 * @func MSPIN_GetServerOSName
 *
 * get name of operating system of connected smartphone.
 * This can be useful when connected via TCP/IP where we cannot distinguish from the USB connection type.
 *
 * @param instanceHandle The instance handle
 * @param osName The pointer to the buffer of the OS name
 * @param length The length of the buffer. The length must be enough to hold the name and the null termination '\0'.
 * @return error code. MSPIN_SUCCESS on success, \
 *                     MSPIN_ERROR_NOT_IMPLEMENTED in case of Core 1.0.x, \
 *                     MSPIN_ERROR_NULL_HANDLE when the instance is NULL, \
 *                     MSPIN_ERROR_CORE when the core is NULL, \
 *                     MSPIN_ERROR_GENERAL when the buffer is too small, the name is empty or any other error
 */
MSPIN_ERROR MSPIN_GetServerOSName(MSPIN_Instance_t* instanceHandle, char* osName, U32 length);

/**
 * @func MSPIN_KeyEvent
 *
 * sends a key event to the phone
 *
 * @param instanceHandle The instance handle
 * @param key identifies the key
 * @param pressed indicates if the key is pressed or released
 */
void MSPIN_KeyEvent(MSPIN_Instance_t* instanceHandle, MSPIN_KEY key, bool pressed);

/**
 * @func MSPIN_SendPosition
 *
 * sends the position to the phone
 *
 * @param instanceHandle The instance handle
 * @param positionNMEA String with the car's position, NMEA encoded
 * @return error code. MSPIN_SUCCESS on success
 */
MSPIN_ERROR MSPIN_SendPosition(MSPIN_Instance_t* instanceHandle, char* positionNMEA);

/**
 * @func MSPIN_SendCustomString
 *
 * used to send a message string to the phone
 *
 * @param instanceHandle The instance handle
 * @param type type of string content
 * @param msgString the message string
 */
void MSPIN_SendCustomString(MSPIN_Instance_t* instanceHandle, MSPIN_HOSTMSGSTRTYPE type, const char* msgString);

/**
 * @func MSPIN_SendCustomInt
 *
 * used to send some data to the phone
 *
 * @param instanceHandle The instance handle
 * @param type Type of data content
 * @param length Length of data content
 * @param data The data itself
 * @return error code. MSPIN_SUCCESS on success
 */
MSPIN_ERROR MSPIN_SendCustomInt(MSPIN_Instance_t* instanceHandle, MSPIN_HOSTMSGINTTYPE type, U16 length, U8* data);

/**
 * @func MSPIN_EnableCustomKeys
 *
 * Enables one of the custom key profiles. The list of supported keys is described in the definition of \
 * MSPIN_CUSTOMDATA_KEYPROFILE.
 *
 * In order to enable forwarding of custom keys this function must be called with the correct profile selected. In addition \
 * the correct client capability must be set to enable key support (e.g. requires focus control). Compare MSPIN_SetCapabilities.
 *
 * This functions must be called before establishing mySPIN connection to the smartphone.
 *
 * @param instanceHandle The instance handle
 * @param profile The custom key profile to be activated. The list of keys enabled for every profile is described in \
 *                the enum list of MSPIN_CUSTOMDATA_KEYPROFILE.
 * @return error code. MSPIN_SUCCESS on success, \
 *                     MSPIN_ERROR_NOT_IMPLEMENTED before Core 1.2.x
 */
MSPIN_ERROR MSPIN_EnableCustomKeys(MSPIN_Instance_t* instanceHandle, MSPIN_CUSTOMDATA_KEYPROFILE profile);

/**
 * @func MSPIN_EnablePTTCustomKey
 *
 * Enables the PushToTalk (PTT) custom key. This can happen in addition to other custom key profiles
 * set via MSPIN_EnableCustomKeys or not.
 *
 * Enabling PTT custom key is independent from the client capability MSPIN_CLIENTCAPABILITIES_REQUIRESFOCUSCONTROL
 * (compare MSPIN_SetCapabilities).
 *
 * This functions must be called before establishing mySPIN connection to the smartphone.
 *
 * Per default PTT custom key is disabled.
 *
 * @param instanceHandle The instance handle
 * @param enable Set to true to enable PTT custom key. Set to false to disable.
 * @return error code. MSPIN_SUCCESS on success, \
 *                     MSPIN_ERROR_NOT_IMPLEMENTED before Core 1.2.x
 */
MSPIN_ERROR MSPIN_EnablePTTCustomKey(MSPIN_Instance_t* instanceHandle, bool enable);

/**
 * @func MSPIN_SendCustomKey
 *
 * send custom key events. Compare also MSPIN_KeyEvent for standard keys
 *
 * @param instanceHandle The instance handle
 * @param type The key type e.g. press
 * @param code The key code e.g. push to talk button
 * @return error code. MSPIN_SUCCESS on success. Errors are:
 *      - MSPIN_ERROR_CORE when mySPIN IVI Core reported an error
 *      - MSPIN_ERROR_NOT_IMPLEMENTED when not implemented for this Core version
 *      - MSPIN_ERROR_NULL_HANDLE when context is NULL or Core instance
 *      - MSPIN_ERROR_NOT_READY when IVI Core isn't ready
 */
MSPIN_ERROR MSPIN_SendCustomKey(MSPIN_Instance_t* instanceHandle, MSPIN_CUSTOMDATA_KEYTYPE type, MSPIN_CUSTOMDATA_KEYCODE code);

/**
 * @func MSPIN_SetNightMode
 *
 * switch the phone to night mode or back
 *
 * @param instanceHandle The instance handle
 * @param isNight True for night, false for day
 */
void MSPIN_SetNightMode(MSPIN_Instance_t* instanceHandle, bool isNight);

/**
 * @func MSPIN_SetMovingStatus
 *
 * inform the phone if the car is moving
 *
 * @param instanceHandle The instance handle
 * @param isMoving indicates if vehicle is moving or not
 */
void MSPIN_SetMovingStatus(MSPIN_Instance_t* instanceHandle, bool isMoving);

/**
 * @func MSPIN_SetVehicleData
 *
 * send a vehicle date to the smartphone
 *
 * @param instanceHandle The instance handle
 * @param key The vehicle date identifier
 * @param encoding The encoding of the string (UTF8, Latin1, ...)
 * @param jsonString The value of the vehicle date
 * @return error code. MSPIN_SUCCESS on success
 */
MSPIN_ERROR MSPIN_SetVehicleData(MSPIN_Instance_t* instanceHandle, U32 key, MSPIN_STRINGENCODING encoding, char* jsonString);

/**
 * @func MSPIN_SetVehicleDataFilter
 *
 * simulate setting of filter list for vehicle data.
 *
 * ATTENTION: This function is only for testing! Do not use!
 *
 * @param instanceHandle The instance handle
 * @param flag to omit call of callback
 * @param length of and keylist
 * @return error code. MSPIN_SUCCESS on success
 */
MSPIN_ERROR MSPIN_SetVehicleDataFilter(MSPIN_Instance_t* instanceHandle, BOOL callbackOn, U8 length, U32* keyList);

/**
 * @func MSPIN_PhoneCallStatus
 *
 * if mySPIN initiates a phone call (see MSPIN_SetInitiatePhoneCallCallback), the head unit has
 * to provide the actual status by calling MSPIN_PhoneCallStatus
 *
 * @param instanceHandle The instance handle
 * @param status new status of phone call
 */
void MSPIN_PhoneCallStatus(MSPIN_Instance_t* instanceHandle, MSPIN_PHONECALLSTATUS status);

/**
 * @func MSPIN_PumpEvents
 *
 * If mySPIN shall consume the touch events coming directly from Wayland (default) or through the
 * used layermanager implementation, then MSPIN_PumpEvents must be called every few ms to check
 * for user input (touch events) and handle it.
 * Else do not call MSPIN_PumpEvents() - instead it might be necessary to wait a few ms to
 * avoid CPU saturation.
 *
 * @param instanceHandle The instance handle
 * @param timeout The polling timeout in milliseconds. 0 means no timeout, a negative value means infinite
 */
void MSPIN_PumpEvents(MSPIN_Instance_t* instanceHandle, S32 timeout);

/**
 * @func MSPIN_TouchEvents
 *
 * sends an array of touch events to the phone
 * This function is optional; you can use it if you don't want mySPIN to consume the touch events
 * coming directly from Wayland. See also MSPIN_SetWaylandTouch().
 *
 * @param instanceHandle The instance handle
 * @param eventArray Array with TouchEvents (contains usually one event for a normal single touch; two for typical gestures)
 * @param numberOfEvents The number of elements in the eventArray
 */
void MSPIN_TouchEvents(MSPIN_Instance_t* instanceHandle, MSPIN_TouchEvent_t* eventArray, U8 numberOfEvents);

/**
 * @func MSPIN_SuspendConnection
 *
 * suspends the mySPIN connection. This is achieved but do not requesting any frame updates. The
 * USB connection itself is not changed. The connection can be resumed by calling @see MSPIN_ResumeConnection
 *
 * @param instanceHandle The instance handle
 * @return error code. MSPIN_SUCCESS on success, \
 *      MSPIN_ERROR_GENERAL when there is no connection, \
 *      MSPIN_ERROR_NULL_HANDLE when the instance is NULL
 */
MSPIN_ERROR MSPIN_SuspendConnection(MSPIN_Instance_t* instanceHandle);

/**
 * @func MSPIN_ResumeConnection
 *
 * resumes a suspended mySPIN connection. This triggers the frame update requests again. A
 * connection can be suspended by calling @see MSPIN_SuspendConnection
 *
 *
 * @param instanceHandle The instance handle
 * @return error code. MSPIN_SUCCESS on success, \
 *      MSPIN_ERROR_GENERAL when there is no connection, \
 *      MSPIN_ERROR_NULL_HANDLE when the instance is NULL
 */
MSPIN_ERROR MSPIN_ResumeConnection(MSPIN_Instance_t* instanceHandle);

/**
 * @func MSPIN_SetAudioStatus
 *
 * Audio status for a previously requested source identified by requestID
 *
 * @param instanceHandle The instance handle
 * @param requestID ID representing the app/source which requested audio output
 * @param status Status
 * @param reason For future use
 * @return error code. MSPIN_SUCCESS on success
 */
MSPIN_ERROR MSPIN_SetAudioStatus(MSPIN_Instance_t* instanceHandle, U32 requestID,
        MSPIN_AUDIOSTATUS status, MSPIN_AUDIORESULTCODE reason);

/**
 * @func MSPIN_CreateAppInfoList
 *
 * Retrieve a list with the available apps from phone.
 * Only one list can be created - before creating a new one,
 * release the previous one with MSPIN_DeleteAppInfoList().
 * Do not change or free any item of the list (this is done by mySPIN IVI Core).
 *
 * @param instanceHandle The instance handle
 * @param numberOfItems Filled with the number of items on return
 * @param infoList Filled with the app info list on return
 * @return error code. MSPIN_SUCCESS on success
 */
MSPIN_ERROR MSPIN_CreateAppInfoList(MSPIN_Instance_t* instanceHandle, U32* numberOfItems, MSPIN_AppInfoList_t** infoList);

/**
 * @func MSPIN_DeleteAppInfoList
 *
 * Deletes the app info list that was created with MSPIN_CreateAppInfoList()
 *
 * @param instanceHandle The instance handle
 * @param infoList The app info list
 * @return error code. MSPIN_SUCCESS on success
 */
MSPIN_ERROR MSPIN_DeleteAppInfoList(MSPIN_Instance_t* instanceHandle, MSPIN_AppInfoList_t* infoList);

/**
 * @func MSPIN_RequestAppIcon
 *
 * Request an icon for an app
 *
 * @param instanceHandle The instance handle
 * @param identifierURL The string that identifies the app
 * @param width The width to which the icon should be scaled by the phone
 * @param height The height to which the icon should be scaled by the phone
 * @return error code. MSPIN_SUCCESS on success
 */
MSPIN_ERROR MSPIN_RequestAppIcon(MSPIN_Instance_t* instanceHandle, const char* identifierURL,
        U16 width, U16 height);

/**
 * @func MSPIN_StartApp
 *
 * Start the App specified by identifierURL.
 *
 * @param instanceHandle The instance handle
 * @param identifierURL The string that identifies the app
 * @param launchModes Key-value pair list containing launch modes
 * @param options Key-value pair list containing options
 * @return error code. MSPIN_SUCCESS on success
 */
MSPIN_ERROR MSPIN_StartApp(MSPIN_Instance_t* instanceHandle, const char* identifierURL,
        MSPIN_KeyValueList_t* launchModes, MSPIN_KeyValueList_t* options);

// Misc:

/**
 * @func MSPIN_SetVerbosity
 *
 * can be used to change the verbosity level for certain logging messages. If modifying verbosity level this
 * should be ideally called before MSPIN_Init(...)
 *
 * Note: DLT or TTFis log levels cannot be adjusted with this function. They will be adjusted via trace commands.
 *
 * ATTENTION: This function is intended for internal use only. Do not change default logging level and releasing
 *            it to project.
 *
 * @param verbosityLevel The verbosity level (a higher value means more messages, range is 0..7)
 * @param prependTime Prepend timestamp Prepend a timestamp. The timestamp will only be prepended when using stdout or
 *                    file as output destination (not for DLT or TTFis)
 */
void MSPIN_SetVerbosity(S32 verbosityLevel, bool prependTime);

/**
 * @func MSPIN_SetLoggingDestination
 *
 * This function can be used to change default logging behavior. Default logging destination is DLT.
 *
 * Note: In some cases an additional logging destination can be enabled. This depends on the current implementation.
 *
 * ATTENTION: This function is intended for internal use only. Do not change default logging destination and releasing
 *            it to project.
 *
 * @param type Enables/disables the the specified logging destination
 * @param enable true enables specified destination, false disables specified destination
 */
void MSPIN_SetLoggingDestination(MSPIN_LOGGING_DESTINATION type, bool enable);

/**
 * @func MSPIN_GetErrorName
 *
 * can be used to translate an error code into a name
 *
 * @param error The error to be translated
 * @return The human readable name representing the error code
 */
const char* MSPIN_GetErrorName(MSPIN_ERROR error);

#ifdef __cplusplus
}
#endif

#endif /* MSPIN_APPL_IF_ADAPTER_H_ */

